/**
* \file: ConfigHandle.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: Unified SPI
*
* \author: M. Adachi / ADITJ/SW / madachi@jp.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <adit_logging.h>
#include <cmath>
#include "uspi/ConfigHandle.h"

#ifndef DLT_UCH
#define DLT_UCH "UCH"
#endif

LOG_DECLARE_CONTEXT(uspi_common)

namespace adit { namespace uspi
{
using std::istringstream;
using std::pair;
using std::string;
using std::map;
using std::vector;
using std::set;

ConfigHandle::ConfigHandle(void)
{
    LOG_REGISTER_CONTEXT(uspi_common, DLT_UCH, "USPI Config handle");
}

ConfigHandle::~ConfigHandle(void)
{
    mConfiguration.clear();
}

ConfigHandle::Range::Range(int inMin, int inMax)
{
    if (inMin > inMax) {
        LOG_ERROR((uspi_common, "invalid range(inMin=%d,inMax=%d)", inMin, inMax));
    }
    min = inMin;
    max = inMax;
}

bool ConfigHandle::Range::validate(int inValue) const
{
    if ((inValue < min) || (inValue > max)) {
        LOG_ERROR((uspi_common, "value out of range(inMin=%d,inMax=%d) inValue=%d", min, max, inValue));
        return false;
    }
    return true;
}

bool ConfigHandle::Range::validate(float inValue) const
{
    if ((floor(inValue) < min) || (ceil(inValue) > max)) {
        LOG_ERROR((uspi_common, "value out of range(inMin=%d,inMax=%d) inValue=%f", min, max, inValue));
        return false;
    }
    return true;
}

bool ConfigHandle::Range::validate(string inValue) const
{
    if (inValue.empty()) {
        LOG_ERROR((uspi_common, "value out of range(string is empty)"));
        return false;
    }
    return true;
}

bool ConfigHandle::Range::validate(void) const
{
    // used as ConfigHandle::novalidator()
    return true;
}

ConfigHandle::Match::Match(int inArgNum, ...)
{
    va_list args;
    va_start(args, inArgNum);

    for(int count = 0; count < inArgNum ; count++)
    {
        params.insert(va_arg(args, int));
    }
    va_end(args);
}

bool ConfigHandle::Match::validate(int inValue) const
{
    bool ret = true;

    if(params.find(inValue) == params.end())
    {
        ret = false;
        LOG_ERROR((uspi_common, "invalid inValue %d", inValue));
    }

    return ret;
}

bool ConfigHandle::Match::validate(float inValue) const
{
    bool ret = true;

    if(params.find(inValue) == params.end())
    {
        ret = false;
        LOG_ERROR((uspi_common, "invalid inValue %f", inValue));
    }

    return ret;
}

bool ConfigHandle::Match::validate(string inValue) const
{
    inValue = inValue;
    return true;
}

bool ConfigHandle::Match::validate(void) const
{
    // used as ConfigHandle::novalidator()
    return true;
}

ConfigHandle::MatchStrings::MatchStrings(int inArgNum, ...)
{
    va_list args;
    va_start(args, inArgNum);

    for(int count = 0; count < inArgNum ; count++)
    {
        string tmp(va_arg(args, const char*));
        strings.insert(tmp);
    }
    va_end(args);
}

bool ConfigHandle::MatchStrings::validate(int inValue) const
{
    inValue = inValue;
    return true;
}

bool ConfigHandle::MatchStrings::validate(float inValue) const
{
    inValue = inValue;
    return true;
}

bool ConfigHandle::MatchStrings::validate(string inValue) const
{
    bool ret = true;

    if(strings.find(inValue) == strings.end())
    {
        ret = false;
        LOG_ERROR((uspi_common, "invalid inValue %s", inValue.c_str()));
    }

    return ret;
}

bool ConfigHandle::MatchStrings::validate(void) const
{
    // used as ConfigHandle::novalidator()
    return true;
}

int ConfigHandle::set(const string& inKey, const string& inValue)
{
    int has_changed = 0;
    if (mConfiguration.size() > 0) {
        auto found = mConfiguration.find(inKey);
        if (found != mConfiguration.end()) {
            if (found->second.compare(inValue) != 0)
            {
                mConfiguration.erase(found);
                mConfiguration.insert(make_pair(inKey, inValue));
                has_changed = 1;
            }
        } else {
            mConfiguration.insert(make_pair(inKey, inValue));
            has_changed = 1;
        }
    } else {
        mConfiguration.insert(make_pair(inKey, inValue));
        has_changed = 1;
    }

   return has_changed;
}

bool ConfigHandle::find(const string& inKey, const string& inSubKey, string& outStr)
{
    bool    ret = false;
    string  retStr;

    if (find(inKey, retStr)) {
        if (findSub(retStr, inSubKey, outStr)) {
            ret = true;
        }
    }

    return ret;
}

bool ConfigHandle::find(const string& inKey, string& outStr)
{
    bool    ret = true;

    auto itrMap = mConfiguration.find(inKey);
    if (itrMap != mConfiguration.end()) {
        outStr = itrMap->second.c_str();
    } else {
        // TODO: should we trace to find a mis-configuration (error or warning)?
        ret = false;
    }

    return ret;
}

// Gets strings of configuration with default value
string ConfigHandle::getString(const string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator, string defaultValue)
{
    string outStr;
    string retVal("");

    if(find(inKey, outStr))
    {
        istringstream stream(outStr);
        retVal = stream.str();

        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_INFO((uspi_common, "couldn't find key=%s", inKey.c_str()));
        retVal = defaultValue;
    }

    return retVal;
}

// Gets strings of configuration
string ConfigHandle::getString(const string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator)
{
    string outStr;
    string retVal("");

    if (find(inKey, outStr)) {
        istringstream stream(outStr);
        retVal = stream.str();

        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_ERROR((uspi_common, "couldn't find key=%s", inKey.c_str()));
        if (outIsValid != nullptr) {
            *outIsValid = false;
        } else {
            LOG_INFO((uspi_common, "Parameter outIsValid is invalid(nullptr)"));
        }
    }

    return retVal;
}

// Gets values of configuration with default value
int ConfigHandle::getInt(const string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator, int defaultValue)
{
    string outStr;
    int retVal = -1;

    if (find(inKey, outStr)) {
        istringstream stream(outStr);
        stream >> retVal;

        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_INFO((uspi_common, "couldn't find key=%s", inKey.c_str()));
        retVal = defaultValue;
    }

    return retVal;
}

// Gets values of configuration
int ConfigHandle::getInt(const string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator)
{
    string outStr;
    int retVal = -1;

    if (find(inKey, outStr)) {
        istringstream stream(outStr);
        stream >> retVal;

        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_ERROR((uspi_common, "couldn't find key=%s", inKey.c_str()));
        if (outIsValid != nullptr) {
            *outIsValid = false;
        } else {
            LOG_ERROR((uspi_common, "Parameter outIsValid is invalid(nullptr)"));
        }
    }

    return retVal;
}

float ConfigHandle::getFloat(const string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator,
                   float defaultValue)
{
    string outStr;
    float retVal = -1;

    if (find(inKey, outStr)) {
        istringstream stream(outStr);
        stream >> retVal;

        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_INFO((uspi_common, "couldn't find key=%s", inKey.c_str()));
        retVal = defaultValue;
    }

    return retVal;
}

float ConfigHandle::getFloat(const string& inKey, bool* outIsValid, const ConfigHandle::Validator& inValidator)
{
    string outStr;
    float retVal = -1;

    if (find(inKey, outStr)) {
        istringstream stream(outStr);
        stream >> retVal;

        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_ERROR((uspi_common, "couldn't find key=%s", inKey.c_str()));
        if (outIsValid != nullptr) {
            *outIsValid = false;
        } else {
            LOG_ERROR((uspi_common, "Parameter outIsValid is invalid(nullptr)"));
        }
    }

    return retVal;
}

int ConfigHandle::getIntMultiKey(const string& inKey, const string& inSubkey, bool* outIsValid,
        const ConfigHandle::Validator& inValidator, int defaultValue)
{
    string outStr;
    int retVal = -1;

    if (find(inKey, inSubkey, outStr)) {
        istringstream stream(outStr);
        stream >> retVal;
        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOGD_DEBUG((uspi_common, "couldn't find key=%s, subkey=%s", inKey.c_str(), inSubkey.c_str()));
        retVal = defaultValue;
    }

    return retVal;
}

int ConfigHandle::getIntMultiKey(const string& inKey, const string& inSubkey, bool* outIsValid,
        const ConfigHandle::Validator& inValidator)
{
    string outStr;
    int retVal = -1;

    if (find(inKey, inSubkey, outStr)) {
        istringstream stream(outStr);
        stream >> retVal;
        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_ERROR((uspi_common, "couldn't find key=%s subkey=%s", inKey.c_str(), inSubkey.c_str()));
        if (outIsValid != nullptr) {
            *outIsValid = false;
        } else {
            LOG_ERROR((uspi_common, "Parameter outIsValid is invalid(nullptr)"));
        }
    }

    return retVal;
}

string ConfigHandle::getStringMultiKey(const string& inKey, const string& inSubkey, bool* outIsValid,
        const ConfigHandle::Validator& inValidator, string defaultValue)
{
    string outStr;
    string retVal("");

    if (find(inKey, inSubkey, outStr)) {
        istringstream stream(outStr);
        retVal = stream.str();
        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOGD_DEBUG((uspi_common, "couldn't find key=%s subkey=%s", inKey.c_str(), inSubkey.c_str()));
        retVal = defaultValue;
    }

    return retVal;
}

string ConfigHandle::getStringMultiKey(const string& inKey, const string& inSubkey, bool* outIsValid,
        const ConfigHandle::Validator& inValidator)
{
    string outStr;
    string retVal("");

    if (find(inKey, inSubkey, outStr)) {
        istringstream stream(outStr);
        retVal = stream.str();
        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_ERROR((uspi_common, "couldn't find key=%s subkey=%s", inKey.c_str(), inSubkey.c_str()));
        if (outIsValid != nullptr) {
            *outIsValid = false;
        } else {
            LOG_ERROR((uspi_common, "Parameter outIsValid is invalid(nullptr)"));
        }
    }

    return retVal;
}

float ConfigHandle::getFloatMultiKey(const string& inKey, const string& inSubkey, bool* outIsValid,
        const ConfigHandle::Validator& inValidator, float defaultValue)
{
    string outStr;
    float retVal = -1;

    if (find(inKey, inSubkey, outStr)) {
        istringstream stream(outStr);
        stream >> retVal;
        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOGD_DEBUG((uspi_common, "couldn't find key=%s, subkey=%s", inKey.c_str(), inSubkey.c_str()));
        retVal = defaultValue;
    }

    return retVal;
}

float ConfigHandle::getFloatMultiKey(const string& inKey, const string& inSubkey, bool* outIsValid,
        const ConfigHandle::Validator& inValidator)
{
    string outStr;
    float retVal = -1;

    if (find(inKey, inSubkey, outStr)) {
        istringstream stream(outStr);
        stream >> retVal;
        bool isValid = inValidator.validate(retVal);
        if (!isValid && outIsValid != nullptr) {
            *outIsValid = false;
            LOG_ERROR((uspi_common, "invalid key=%s", inKey.c_str()));
        }
    } else {
        LOG_ERROR((uspi_common, "couldn't find key=%s subkey=%s", inKey.c_str(), inSubkey.c_str()));
        if (outIsValid != nullptr) {
            *outIsValid = false;
        } else {
            LOG_ERROR((uspi_common, "Parameter outIsValid is invalid(nullptr)"));
        }
    }

    return retVal;
}

bool ConfigHandle::findSub(const string& inSrcString, const string& inFindString, string& outDstString)
{
    bool ret = false;
    map < string, string > settingsMap = parseMap(inSrcString, true);

    // note, parsed keys are converted to small letters
    auto iter = settingsMap.find(inFindString);
    if (iter != settingsMap.end())
    {
        ret = true;
        outDstString = iter->second;
    }
    else
    {
        // TODO: should we trace to find a mis-configuration (error or warning)?
        outDstString = "";
    }

    return ret;
}

vector<string> ConfigHandle::splitString(const string& inValue, const char inSeparators[])
{
    vector<string> result;

    string::size_type current = 0;
    string::size_type found;
    while (string::npos != (found = inValue.find_first_of(inSeparators, current)))
    {
        result.emplace_back(inValue, current, found - current);
        current = found + 1;
    }

    result.emplace_back(inValue, current);
    return result;
}

string ConfigHandle::trimString(const string& inValue)
{
    string::size_type begin = inValue.find_first_not_of(" \t");
    if (begin == string::npos)
        begin = 0;
    string::size_type length = inValue.find_last_not_of(" \t");
    if (length == string::npos)
        length = inValue.length() - 1;
    else
        length = length - begin + 1;
    return inValue.substr(begin, length);
}

map<string, string> ConfigHandle::parseMap(const string& inValue, bool inLowerCase)
{
    map<string, string> dictionary;

    // split the string into 'xx=yy' key-value pairs
    vector<string> keyValuePairs = splitString(inValue, " \t,");
    for (auto keyValuePair : keyValuePairs)
    {
        // split into key and value
        vector<string> keyValueVector = splitString(trimString(keyValuePair), "=");

        // in case we have multiple or no '='
        if (keyValueVector.size() != 2)
        {
            LOG_ERROR((uspi_common, "could not parse %s as key-value pair", trimString(keyValuePair).c_str()));
        }
        // in case we have a clear key-value pair
        else
        {
            string key = trimString(keyValueVector[0]);
            if (inLowerCase)
            {
                std::transform(key.begin(), key.end(), key.begin(),
                        std::ptr_fun<int, int>(std::tolower));
            }
            string value = trimString(keyValueVector[1]);

            // insert into result map
            dictionary.insert(pair<string, string>(key, value));
        }
    }

    return dictionary;
}

} } /* namespace adit { namespace uspi { */


